/************************************************************************************
Program Name: MultiDXCCS.do 
Description : Assign Multi-Level Diagnosis CCS category arrays (L1dccs# L2dccs# L3dccs# L4dccs#) from Stata. 
Developed   : By Bob Houchens
Updated     : By David Ross on 10/27/2009.
            : By David Ross and Nils Nordstrand on 11/03/2016 incorporating updates from Rebecca Pope (StataCorp).
************************************************************************************/

version 9

local modern = c(stata_version) >= 14
local old = c(stata_version) < 11

local vv : di "version " string(_caller()) ":"
*
unab varlist: DX*
local _ndx: word count `varlist'
*
// only reshape if there is more than one DX 
if `_ndx' > 1 {
    tempvar i j
    gen `i' = _n 	// use -generate- instead of -egen- (faster)
    qui compress	// reshape is slow, but faster the smaller the dataset is
    reshape long DX, i(`i') j(`j') 
}
*
* Generate a temporary diagnosis variable that will be reformatted by the clean function in preparation for the merge
*
generate _DX = DX
*
* Check the validity of the diagnosis code
*
tempvar invalid
capture icd9 check _DX, generate(`invalid')
*
* Format the temporary diagnosis to match the format in multiDXCCS.dta  
*
if `modern' {
    `vv' qui icd9 clean _DX, dots, if `invalid'==0 | `invalid'==10
}
*
* replace invalid temporary diagnoses in preparation for the clean function
* (only necessary before Stata 14)
*
else {
    qui replace _DX="0000" if `invalid' > 0 & `invalid' < 10
    qui icd9 clean _DX, dots
}
drop `invalid'
*
* Merge the multi-level CCS category variables (L1dccs L2dccs L3dccs L4dccs) that match the temporary diagnosis
*
if `old' {
    * Sort by formatted diagnosis.
    sort _DX
    merge _DX using multiDXCCS, nokeep
    drop _merge _DX
}
else {
    `vv' merge m:1 _DX using multiDXCCS, keep(1 3) nogen 
    capture drop _DX
}
*
* Drop temporary variables and put data in original shape if needed
*
if `_ndx' > 1 {
    reshape wide DX L1dccs L2dccs L3dccs L4dccs, i(`i') j(`j')
    capture drop `i' 
}
*
exit

